﻿//
// 3D物理エンジン Open Dynamics Engine を用いた1輪車のプログラム
// Open Dynamics Engine 本家: http://www.ode.org/
// Open Dynamics Engine Wiki: http://opende.sourceforge.net/wiki/index.php/Main_Page
// 参考サイト1: http://www.crystal-creation.com/
// 参考サイト2: http://demura.net/
// 参考サイト3: http://www.koj-m.sakura.ne.jp/ode/index.php?Open%20Dynamics%20Engine
// 参考サイト4: http://hwm7.gyao.ne.jp/shinpuku/index.html
// 参考サイト5: http://www10.atwiki.jp/bambooflow/
//

//
// GNU Scientific Library
//
// 本家: http://www.gnu.org/software/gsl/
// 参考サイト1: http://www.tmps.org/index.php?TMPSwiki
// 参考サイト2: http://gonzaburou.cocolog-nifty.com/blog/2006/11/gslvisual_cc_8716.html
//

#ifndef MAIN_H // 二重インクルードを防止
#define MAIN_H

//
// for Open Dynamics Engine
//
#include <ode/ode.h>             // ODEを使うことを開発環境にお知らせ
#include <drawstuff/drawstuff.h> // 画面表示を受け持つ部分

#define DEBUG 1     // 1:デバッグを有効 0:デバッグを無効
#define SAMPLINGTIME (50.0) // [msec]で制御対象の状態をサンプリング
#define GRAVITY (-9.8) // -9.8[m/{sec}^2]
#define MAX_CONTACTS (25) // 衝突検出の精度: 大きいほど精度が上がるが、計算コストがかかる
#define DT  ((double)((double)(SAMPLINGTIME) / (1000.0))) // ODEの世界での物理演算の刻み時間 dt[sec] ***この値を小さくするほど精度は上がるが計算時間がかかる***
//#define FPS 60.0 // [frame/sec] 1秒間に何回画面を更新するか
//#define SPF (1.0 / FPS) // [sec/frame] 1フレームに何秒要するか

//
// for GNU Scientific Library
//
#include "Vector.h"                  // ベクトルを扱うC++でラッパークラス
#include "Matrix.h"                  // 行列を扱うC++でラッパークラス
#pragma comment(lib, "gsl.lib")      // GSLを使用するためのスタティックリンクライブラリ
#pragma comment(lib, "gslcblas.lib") // GSLを使用するためのスタティックリンクライブラリ

//
// Open Dynamics Engineを使用する時、Windows上で精度を補償するためのおまじない
//
#ifdef _MSC_VER
#pragma warning(disable:4244 4305)  // for VC++, no precision loss complaints
#endif


#include <stdio.h>   /* c言語 標準入出力 */
#include <stdlib.h>  /* c言語 基本関数を含む */
#include <string.h>  /* c言語 文字列操作ライブラリ */
#include <math.h>    /* c言語 数学ライブラリ */
#include <time.h>    /* c言語 時間の扱い */

#include <iostream> // c++ 標準入出力
#include <iomanip>  // c++ 標準入出力マニピュレータ
#include <fstream>  // c++ ファイルハンドリン

//
// Open Dynamics Engine の内部仕様の古い関数を新しい関数で暗黙的に置き換え
//
#ifdef dDOUBLE
#define dsDrawSphere  dsDrawSphereD
#define dsDrawBox     dsDrawBoxD
#define dsDrawCylinder dsDrawCylinderD
#define dsDrawCapsule  dsDrawCapsuleD
#define dsDrawLine     dsDrawLineD
#define dsDrawTriangle dsDrawTriangleD
#endif

//
// 物体の情報を格納する構造体
//
typedef struct {
  dBodyID body;                        // ボディのID番号(描画と物体の存在を格納)
  dGeomID geom;                        // ジオメトリのID番号(衝突計算に使う幾何情報)
} MyObject;                            // MyObject構造体

enum
{
	SingleWheelTypeModel = 1, // 1輪車
	InertiaRotorPendulum, // 慣性ロータ型振り子
};

enum
{
	LearningControl = 1, // 学習制御
	InitialValueResponse, // 初期値応答
	AdaptiveControl, // 適応制御
};

//
// 新しい変数
//
int RunFlag; // どの制御対象を動かすか
int ControlFlag; // 学習制御、初期値応答、適応制御のどれか
int LearningLength; // 現在の学習回数、クロック
int LearningLengthFinal; // 最終学習回数
int Dimension; // 使用する状態変数の次数 (例)[x0, x1, x2] --> 3次 旧 DD
int NumberOfInput; // 使用する入力数
int BlockLength; // 現在のブロック長さ、多段決定に用いられる区間
int BlockLengthFinal; // ブロック長の終端長さ
int RunTime; // [msec] 実行時間

double Sigma; // 追加入力雑音の振幅 σ
double Delta; // 逆行列補題の初期値 δ 小さいほど理論に忠実となるが計算機の扱える桁に注意する
double Lambda1; // λ1 [0<=1] ゲイン更新に関わる学習係数
double Lambda2; // λ2 [0<=1] ゲイン更新に関わる入力係数

CMatrix x; // 状態変数 x(=sensor value)
CMatrix z; // 応答信号 z
CMatrix u; // 制御入力 u(=input value, etc voltage, motor torque ...)
CMatrix v; // 追加入力雑音 v 入力数と同じ数のベクトル、各雑音は独立して生成
CMatrix xi; // 識別信号 ξ
CMatrix C; // 状態変数の重み行列 C
CMatrix D; // 入力の重み行列 D
CMatrix H; // 不確実な動作に対する統計処理結果を格納する行列 H
CMatrix P; // リカッチ方程式の解
CMatrix Gamma; // 感度計算に用いる行列 Γ
CMatrix Gamma_aa; // 感度計算に用いる行列 Γaa
CMatrix Gamma_ab; // 感度計算に用いる行列 Γab
CMatrix Gamma_ba; // 感度計算に用いる行列 Γba
CMatrix Gamma_bb; // 感度計算に用いる行列 Γbb
CMatrix Psi; // 感度計算に用いる行列 Ψ
CMatrix Upsilon; // 感度計算に用いる行列 Υ
CMatrix Xi; // 感度計算に用いる行列 Ξ
CMatrix G; // ゲイン G(=Feedback gain, etc k ...)

CMatrix I; // 単位行列I

// 後で切り離すか考える
CMatrix A;
CMatrix B;

// ---------------
// 共通するモノ
// ---------------
dWorldID world;  // 動力学の世界
dSpaceID space;  // 衝突検出用スペース
dGeomID  ground; // 地面
dJointGroupID contactgroup;  // ジョイントグループ(物体と地面との衝突に使用)
dsFunctions fn;  // ODEに用意されている関数のエントリを格納

// -----------------
// ジョイントの定義(回転軸に使用)
// -----------------
#define SingleWheelTypeModelJointMax 1
dJointID SingleWheelTypeModelJoint[SingleWheelTypeModelJointMax];

#define InertiaRotorPendulumJointMax 3
dJointID InertiaRotorPendulumJoint[InertiaRotorPendulumJointMax];


// -------------
// 物体の定義
// -------------
/* 車輪重量 8.0kg 車輪半径 0.43m 車輪厚み 0.35m */
#define WHEEL_MASS 10.0 // 車輪重量
#define WHEEL_RADIUS 0.40 // 半径
#define WHEEL_LENGTH 0.80 // 厚み
MyObject wheel; // 車輪

/* 本体重量 10.0kg 本体X 0.14m 本体Y 0.4m 本体Z 1.0m */
#define FRAME_MASS 60.0 // 本体重量
#define FRAME_X 0.14 // 
#define FRAME_Y 0.85 //
#define FRAME_Z 1.2 //
MyObject frame; // フレーム

/* 慣性ロータの重量 2.0kg 慣性ロータの半径 0.30m 慣性ロータの厚み 0.03m */
#define ROTOR_MASS 20.0 // 慣性ロータの重量
#define ROTOR_RADIUS 0.35 // 慣性ロータの半径
#define ROTOR_LENGTH 0.03 // 慣性ロータの厚み
MyObject rotor; // 慣性ロータ

/* 振り子の重量 10.0kg 振り子のX 0.1m 振り子のY 0.1m 振り子のZ 1.0m */
#define PENDULUM_MASS 60.0
#define PENDULUM_X 0.1
#define PENDULUM_Y 0.1
#define PENDULUM_Z 1.2
MyObject pendulum; // 振り子

/* スタンドの重量 100.0kg スタンドのX 0.3m スタンドのY 0.3m スタンドのZ 0.3m */
#define STAND_MASS 50.0
#define STAND_X 0.3
#define STAND_Y 0.3
#define STAND_Z 0.3
MyObject stand; // 固定台

void simLoop(int pause); // ODEメイン
void start(); // ODE開始時に呼ばれる関数
void setDrawStuff(); // 描画を受け持つ関数
void nearCallback(void *data, dGeomID o1, dGeomID o2); // 衝突検出用コールバック関数
void stop(); // ODE停止時に呼ばれる関数
void command(); // ODEキーイベント時に呼ばれる関数
void restart(); // ODE再スタート時に呼ばれる関数
void InitALL(); // ODEの初期化関数

void SingleWheelTypeModelCreateFrame(); // 本体を作成する関数
void SingleWheelTypeModelCreateWheel(); // 車輪を作成する関数
void SingleWheelTypeModelCreateJointFrameAndWheel(); // 1輪車の回転軸を作成する関数

void InertiaRotorPendulumCreateRotor(); // 慣性ロータを作成する関数
void InertiaRotorPendulumCreatePendulum(); // 慣性ロータの振り子を作成する関数
void InertiaRotorPendulumCreateStand(); // 慣性ロータのスタンドを作成する関数
void InertiaRotorPendulumCreateJointStand(); // 台の固定ジョイントを作成
void InertiaRotorPendulumCreateJointStandAndPendulum(); // 台と振り子のジョイントを作成

//------------------------------------------------------
// 関数の広域宣言(どのc/cppコードからも呼び出し可能)
//------------------------------------------------------
extern void simLoop(int pause); // メインシミュレーションループ
extern void start(); // プログラム開始時に呼ばれる関数
extern void setDrawStuff(); // 描画を受け持つ関数
extern void nearCallback(void *data, dGeomID o1, dGeomID o2); // 衝突検出用コールバック関数
extern void stop(); // ODE停止時に呼ばれる関数
extern void command(); // ODEキーイベント時に呼ばれる関数
extern void restart(); // ODE再スタート時に呼ばれる関数
extern void InitALL(); // ODEの初期化関数

extern void GetRollPitchYaw(dBodyID tmpbody, dReal *rpy, const dReal *angle_rpy); // 指定された物体のロー角、ピッチ角、ヨー角およびその角速度を取得する関数
extern void dJointGetHingeAngleInfiniteDiff(dJointID jointID, double *Angle, double dt); // dJointGetHingeAngleの-∞<-->+∞版
extern void SingleWheelTypeModelCreateFrame(); // 本体を作成する関数
extern void SingleWheelTypeModelCreateWheel(); // 車輪を作成する関数
extern void SingleWheelTypeModelCreateJointFrameAndWheel(); // 1輪車の回転軸を作成する関数

extern void InertiaRotorPendulumCreateRotor(); // 慣性ロータを作成する関数
extern void InertiaRotorPendulumCreatePendulum(); // 慣性ロータの振り子を作成する関数
extern void InertiaRotorPendulumCreateStand(); // 慣性ロータのスタンドを作成する関数
extern void InertiaRotorPendulumCreateJointStand(); // 台の固定ジョイントを作成

using namespace std; // std名前空間を省略

//
// プロットデータファイル
//
ofstream state_file ("./data/state.dat"); // 状態データ
ofstream state_csv_file ("./data/state.csv"); // 状態データ
ofstream gain_file ("./data/gain.dat"); // ゲインデータ
ofstream gain_csv_file ("./data/gain.csv"); // 状態データ
ofstream input_file ("./data/input.dat"); // 入力データ
ofstream input_csv_file ("./data/input.csv"); // 入力データ
ofstream rpm_file ("./data/rpm.dat"); // 回転数データ

void PrintMatrix(const char *name, const CMatrix &m); // 指定された行列のコンソールへの表示
void MatrixTest(void); // 行列の数値テスト

//
// 一輪車関係
//
dReal frame_rpy[3] = {0.0, 0.0, 0.0}; // 本体のrpy[0] = ロール角、rpy[1] = ピッチ角、rpy[2] = ヨー角
dReal frame_angle_rpy[3] = {0.0, 0.0, 0.0}; // 本体のangle_rpy[0] = ロール角速度、angle_rpy[1] = ピッチ角速度、angle_rpy[2] = ヨー角速度
dReal wheel_rpy[3] = {0.0, 0.0, 0.0}; // 車輪の rpy[0] = ロール角、rpy[1] = ピッチ角、rpy[2] = ヨー角
dReal wheel_angle_rpy[3] = {0.0, 0.0, 0.0}; // 車輪のangle_rpy[0] = ロール角速度、angle_rpy[1] = ピッチ角速度、angle_rpy[2] = ヨー角速度

dReal prev_wheel_pitch = 0.0; // 過去のホイールのピッチ角
dReal prev_frame_pitch = 0.0; // 過去の本体のピッチ角
//
//
//
dReal pendulum_rpy[3] = {0.0, 0.0, 0.0};
dReal pendulum_angle_rpy[3] = {0.0, 0.0, 0.0};
dReal prev_pendulum_roll = 0.0;

static struct dStopwatch mytimer; // ODE内のタイマー
double realtime = 0.0; // 実時間
double phystime = 0.0; // 物理演算空間内時間
double drawtime = 0.0; // 描画ウェイトのための時間

void GammaUpdateFromGamma_aa_ab_ba_bb(void); // ΓからΓaa、Γab、Γba、Γbbを得る
void Gamma_aa_ab_ba_bbUpdateFromGamma(void); // Γaa、Γab、Γba、ΓbbからΓを得る
void zUpdate(void); // 応答信号zの更新
void xiUpdate(void); // 識別信号ξの更新
void RecordGraphx(void); // 状態のグラフ出力
void RecordGraphu(void); // 入力のグラフ出力
void RecordGraphG(void); // ゲインのグラフ出力
void GainUpdate(void); // ゲインに関わる更新
void SensitivityMain(void); // 感度計算
void Control(void); // 制御入力を決定
void ResetModel(void); // 制御対象のモデルをリセット
void InitializeAllParameter(void); // 理論の全パラメータを初期化

double InertiaRotorAngle;

#endif